#include "HX711.h"
#include <Wire.h>
#include <U8g2lib.h>
#include <WiFi.h>
#include <PubSubClient.h>

// 接線設定
const int DT_PIN = 4;
const int SCK_PIN = 2;
const int scale_factor = 353.179; // 比例參數，從校正程式中取得

// OLED 顯示屏設定
const int OLED_SCL_PIN = 7;
const int OLED_SDA_PIN = 8;

char* ssid = "xxxxx";     //wifi名稱
char* password = "xxxxxxx";  //wifi 密碼

const char* mqttServer = "broker.hivemq.com";  // MQTT伺服器位址
const char* mqttUserName = "xxxxxx";  // 使用者名稱，隨意設定。
const char* mqttPwd = "xxxxxx";     // MQTT密碼
const char* clientID = "xxxxxx";      // 用戶端ID，隨意設定。
const char* topic = "xxxxxx";       //MQTT 伺服器名稱

unsigned long prevMillis = 0;  // 暫存經過時間（毫秒）
const long interval = 5000;  // 上傳資料的間隔時間，5秒。
String msgStr = "";      // 暫存MQTT訊息字串

HX711 scale;
U8G2_SSD1306_128X64_NONAME_F_HW_I2C u8g2(U8G2_R0, /* reset=*/ U8X8_PIN_NONE, OLED_SCL_PIN, OLED_SDA_PIN);

WiFiClient BW16Client;
PubSubClient client(BW16Client);

void setup_wifi() {
  delay(10);

  WiFi.begin(ssid, password);

  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(F("."));
  }

  Serial.println("");
  Serial.println(F("WiFi connected"));
}

void reconnect() {
  while (!client.connected()) {
    if (client.connect(clientID, mqttUserName, mqttPwd)) {
      Serial.println(F("MQTT connected"));
    } else {
      Serial.print(F("failed, rc="));
      Serial.print(client.state());
      Serial.println(F(" try again in 5 seconds"));
      delay(5000);  // 等5秒之後再重試
    }
  }
}

void setup() {
  Serial.begin(9600);
  Serial.println(F("Initializing the scale"));

  scale.begin(DT_PIN, SCK_PIN);

  Serial.println(F("Before setting up the scale:"));
  Serial.println(scale.get_units(5), 0);  // 未設定比例參數前的數值

  scale.set_scale(scale_factor);  // 設定比例參數
  scale.tare();  // 歸零

  Serial.println(F("After setting up the scale:"));
  Serial.println(scale.get_units(5), 0);  // 設定比例參數後的數值

  Serial.println(F("Readings:"));  // 在這個訊息之前都不要放東西在電子秤上

  // 初始化OLED顯示屏
  u8g2.begin();
  u8g2.enableUTF8Print(); // 允許UTF-8字元
  u8g2.clearBuffer(); // 清除緩衝區
  u8g2.setFont(u8g2_font_ncenB18_tr); // 設置字體，字體大小為18
  u8g2.drawStr(0, 20, "Scale Initialized"); // 顯示初始化訊息
  u8g2.sendBuffer(); // 傳送緩衝區內容到顯示屏

  setup_wifi(); // 初始化 WiFi
  client.setServer(mqttServer, 1883); // 初始化 MQTT
}

void loop() {
  if (WiFi.status() != WL_CONNECTED) {
    setup_wifi();
  }

  if (!client.connected()) {
    reconnect();
  }

  client.loop();

  int weight = scale.get_units(10); // 取得10次平均值
  Serial.println(weight, 0);

  // 準備顯示的文字
  char buffer[10];
  snprintf(buffer, sizeof(buffer), "%d g", (int)weight);

  // 計算居中位置
  u8g2_uint_t text_width = u8g2.getStrWidth(buffer);
  u8g2_uint_t x = (128 - text_width) / 2;
  u8g2_uint_t y = (64 / 2) + (18 / 2); // 64是屏幕高度，18是字體高度

  // 在OLED顯示屏上顯示重量
  u8g2.clearBuffer(); // 清除緩衝區
  u8g2.setFont(u8g2_font_ncenB18_tr); // 設置字體
  u8g2.drawStr(x, y, buffer); // 居中顯示重量值
  u8g2.sendBuffer(); // 傳送緩衝區內容到顯示屏

  scale.power_down(); // 進入睡眠模式
  delay(1000);
  scale.power_up(); // 結束睡眠模式

  if (millis() - prevMillis > interval) {
    prevMillis = millis();  // 更新時間戳

    // 轉換重量數據為字串，並組裝JSON訊息
    String payload = "{\"weight\":\"" + String(weight) + "\"}"; // 以兩位小數的格式將重量轉換為字串
    msgStr = payload; // 更新 MQTT 訊息字串

    // 宣告字元陣列
    byte arrSize = msgStr.length() + 1;
    char msg[arrSize];

    Serial.print(F("Publish message: "));
    Serial.println(msgStr);
    msgStr.toCharArray(msg, arrSize); // 把String字串轉換成字元陣列格式
    client.publish(topic, msg);       // 發布MQTT主題與訊息
  }
}
