import paho.mqtt.client as mqtt
import mysql.connector
import json
import os
import sys
import time
from datetime import datetime

# 創建默認配置
default_config = {
    "mqtt": {
        "broker": "your_mqtt_broker",
        "port": 1883,
        "topic": "your/topic"
    },
    "database": {
        "host": "your_db_host",
        "user": "your_db_user",
        "password": "your_db_password",
        "name": "your_db_name"
    }
}


# 讀取或創建配置文件
def load_config():
    config_path = "config.json"
    if not os.path.exists(config_path):
        with open(config_path, 'w') as config_file:
            json.dump(default_config, config_file, indent=4, ensure_ascii=False)
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 找不到配置文件，已創建默認配置文件 {config_path}")
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 請修改配置文件後重啟程式。")
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 程式5秒後自動關閉。")
        time.sleep(5)  # 等待5秒後自動關閉
        sys.exit(1)  # 終止程式並提示用戶重啟

    with open(config_path, 'r') as config_file:
        config = json.load(config_file)

    return config


# 建立資料庫連接並創建表格（如果尚未創建）
def init_db(db_config):
    try:
        conn = mysql.connector.connect(
            host=db_config["host"],
            user=db_config["user"],
            password=db_config["password"],
            database=db_config["name"]
        )
        cursor = conn.cursor()
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS spo2_data (
                id INT AUTO_INCREMENT PRIMARY KEY,
                device_id INT,
                timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
                bpm INT,
                spo2 INT,
                pi FLOAT
            )
        ''')
        conn.commit()
        cursor.close()
        conn.close()
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 資料庫和資料表初始化成功。")
    except mysql.connector.Error as err:
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 錯誤: {err}")


# 當接收到來自MQTT的消息時的回調函數
def on_message(client, userdata, msg):
    try:
        payload = msg.payload.decode("utf-8")
        data = json.loads(payload)
        d = data.get("d", {})
        device_id = d.get("id")
        bpm = d.get("bpm")
        spo2 = d.get("spo2")
        pi = d.get("pi")
        store_data(db_config, device_id, bpm, spo2, pi)
    except Exception as e:
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 處理消息時發生錯誤: {e}")


# 將接收到的數據存入資料庫
def store_data(db_config, device_id, bpm, spo2, pi):
    try:
        conn = mysql.connector.connect(
            host=db_config["host"],
            user=db_config["user"],
            password=db_config["password"],
            database=db_config["name"]
        )
        cursor = conn.cursor()
        debug_message = f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 正在插入資料到 spo2_data：device_id={device_id}, bpm={bpm}, spo2={spo2}, pi={pi}"
        print(debug_message)
        cursor.execute('''
            INSERT INTO spo2_data (device_id, bpm, spo2, pi) VALUES (%s, %s, %s, %s)
        ''', (device_id, bpm, spo2, pi))
        conn.commit()
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 資料插入成功。")
        cursor.close()
        conn.close()
    except mysql.connector.Error as err:
        print(f"{datetime.now().strftime('%Y-%m-%d %H:%M:%S')} - 存儲資料時發生錯誤: {err}")


# 設定MQTT客戶端
def setup_mqtt(mqtt_config):
    client = mqtt.Client(protocol=mqtt.MQTTv311)
    client.on_message = on_message

    client.connect(mqtt_config["broker"], mqtt_config["port"], 60)
    client.subscribe(mqtt_config["topic"])

    client.loop_forever()


if __name__ == "__main__":
    print("「血氧監測」數據上傳資料庫程式啟動")
    config = load_config()
    db_config = config["database"]
    mqtt_config = config["mqtt"]

    init_db(db_config)
    setup_mqtt(mqtt_config)
