import json
from flask import Flask, request, abort
from linebot import LineBotApi, WebhookHandler
from linebot.models import MessageEvent, TextMessage, FlexSendMessage, BubbleContainer, BoxComponent, TextComponent, SeparatorComponent
from linebot.exceptions import InvalidSignatureError
import paho.mqtt.client as mqtt
from datetime import datetime
import os

app = Flask(__name__)

# 設置預設配置
DEFAULT_CONFIG = {
    "line": {
        "channel_access_token": "your_line_channel_access_token_here",
        "channel_secret": "your_line_channel_secret_here"
    },
    "mqtt": {
        "broker": "your_mqtt_broker_address_here",
        "port": "1883",
        "topic": "oximeter/json"
    }
}

# 讀取或創建配置文件
def load_config():
    config_path = 'config.json'
    if not os.path.exists(config_path):
        with open(config_path, 'w') as file:
            json.dump(DEFAULT_CONFIG, file, indent=4)
        config = DEFAULT_CONFIG
    else:
        with open(config_path, 'r') as file:
            config = json.load(file)

    # 確保 MQTT 端口是整數
    config['mqtt']['port'] = int(config['mqtt']['port'])
    return config

config = load_config()

# 你的 Channel Access Token 和 Channel Secret
line_bot_api = LineBotApi(config['line']['channel_access_token'])
handler = WebhookHandler(config['line']['channel_secret'])

# 文件路徑
USER_MAP_FILE = 'user_map.json'

# 儲存ID和LINEID的映射
user_map = {}

# 讀取用戶映射信息
def load_user_map():
    global user_map
    if os.path.exists(USER_MAP_FILE):
        with open(USER_MAP_FILE, 'r') as file:
            user_map = json.load(file)
    else:
        user_map = {}

# 保存用戶信息
def save_user_map():
    with open(USER_MAP_FILE, 'w') as file:
        json.dump(user_map, file)

# 加載用戶資料
load_user_map()

# MQTT 設定
MQTT_BROKER = config['mqtt']['broker']
MQTT_PORT = config['mqtt']['port']
MQTT_TOPIC = config['mqtt']['topic']

# 當收到連接確認時呼叫
def on_connect(client, userdata, flags, rc):
    print(f"已連接，結果代碼: {rc}")
    client.subscribe(MQTT_TOPIC)


# 當收到消息時調用
def on_message(client, userdata, msg):
    try:
        payload = json.loads(msg.payload.decode())
        data = payload['d']
        user_id = str(data['id'])
        spo_value = data['spo2']
        if user_id in user_map:
            line_user_id = user_map[user_id]
            current_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

            if spo_value < 90:
                alert_message = '血氧濃度嚴重低於標準值'
                alert_message1 = '請立即採取措施！'
                alert_color = '#FF0000'  # 紅色
                alt_text = '血氧濃度警告'
                warning_text = '警告！'
            elif 90 <= spo_value <= 94:
                alert_message = '血氧濃度略低'
                alert_message1 = '建議注意休息與呼吸'
                alert_color = '#FF8C00'  # 橙色
                alt_text = '血氧濃度提示'
                warning_text = '注意！'
            else:
                return  # 如果血氧正常，不進行任何操作

            message = FlexSendMessage(
                alt_text=alt_text,
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text=f'⚠️ {warning_text}', weight='bold', size='xl',
                                          color=alert_color, wrap=True,),
                            TextComponent(text=f'{alert_message}', weight='bold', size='xl',
                                          color=alert_color, wrap=True, ),
                            TextComponent(text=f'{alert_message1}', weight='bold', size='xl',
                                          color=alert_color, wrap=True, ),
                            SeparatorComponent(margin='md'),
                            TextComponent(text=f'ID: {user_id}', size='lg', margin='md', weight='bold',
                                          wrap=True, ),
                            TextComponent(text=f'血氧濃度: {spo_value}%', size='lg', margin='md', weight='bold',
                                          wrap=True,),
                            TextComponent(text=f'發送時間: {current_time}', size='lg', margin='md', weight='bold',
                                          wrap=True,)
                        ]
                    )
                )
            )
            line_bot_api.push_message(line_user_id, message)
    except Exception as e:
        print(f"Error: {e}")


# 初始化 MQTT 客戶端
mqtt_client = mqtt.Client()
mqtt_client.on_connect = on_connect
mqtt_client.on_message = on_message
mqtt_client.connect(MQTT_BROKER, MQTT_PORT, 60)

# 啟動 MQTT 客戶端
mqtt_client.loop_start()

@app.route("/callback", methods=['POST'])
def callback():
    # 獲取 X-Line-Signature header value
    signature = request.headers['X-Line-Signature']

    # 獲取 request body 內容
    body = request.get_data(as_text=True)
    app.logger.info("Request body: " + body)

    # 驗證訊息是否合法
    try:
        handler.handle(body, signature)
    except InvalidSignatureError:
        abort(400)

    return 'OK'

@handler.add(MessageEvent, message=TextMessage)
def handle_message(event):
    text = event.message.text.strip()
    line_user_id = event.source.user_id

    if text.startswith('綁定'):
        parts = text.split()
        if len(parts) == 2:
            _, user_id = parts
            if user_id in user_map and user_map[user_id] != line_user_id:
                message = FlexSendMessage(
                    alt_text='綁定失敗',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='❌ 綁定失敗', weight='bold', size='xl', color='#FF0000', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text=f'ID 「{user_id}」 已綁定到另一個LINE帳戶。', size='lg', margin='md',
                                              weight='bold', wrap=True)
                            ]
                        )
                    )
                )
            else:
                user_map[user_id] = line_user_id
                save_user_map()
                message = FlexSendMessage(
                    alt_text='綁定成功',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='✅ 綁定成功！', weight='bold', size='xl', color='#00FF00', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text=f'ID 「{user_id}」 已綁定到您的LINE帳戶。', size='lg', margin='md',
                                              weight='bold', wrap=True)
                            ]
                        )
                    )
                )
        else:
            message = FlexSendMessage(
                alt_text='格式錯誤',
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text='❌ 格式錯誤。', weight='bold', size='xl', color='#FF0000', wrap=True),
                            SeparatorComponent(margin='md'),
                            TextComponent(text='請使用：綁定 <您的ID>。', size='lg', margin='md', weight='bold',wrap=True)
                        ]
                    )
                )
            )

    elif text.startswith('更改ID'):
        parts = text.split()
        if len(parts) == 3:
            _, old_user_id, new_user_id = parts
            if old_user_id in user_map and user_map[old_user_id] == line_user_id:
                user_map.pop(old_user_id)
                user_map[new_user_id] = line_user_id
                save_user_map()
                message = FlexSendMessage(
                    alt_text='更改ID成功',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='✅ 更改ID成功！', weight='bold', size='xl', color='#00FF00', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text=f'ID 「{old_user_id}」 已更改為 「{new_user_id}」。', size='lg', margin='md', weight='bold', wrap=True)
                            ]
                        )
                    )
                )
            else:
                message = FlexSendMessage(
                    alt_text='更改ID失敗',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='❌ 更改ID失敗。', weight='bold', size='xl', color='#FF0000', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text='您輸入的舊ID未綁定到您的LINE帳戶。', size='lg', margin='md', weight='bold', wrap=True)
                            ]
                        )
                    )
                )
        else:
            message = FlexSendMessage(
                alt_text='格式錯誤',
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text='❌ 格式錯誤。', weight='bold', size='xl', color='#FF0000', wrap=True),
                            SeparatorComponent(margin='md'),
                            TextComponent(text='請使用：更改ID <舊的ID> <新的ID>。', size='lg', margin='md', weight='bold', wrap=True)
                        ]
                    )
                )
            )

    elif text.startswith('解除綁定'):
        parts = text.split()
        if len(parts) == 2:
            _, user_id = parts
            if user_id in user_map and user_map[user_id] == line_user_id:
                user_map.pop(user_id)
                save_user_map()
                message = FlexSendMessage(
                    alt_text='解除綁定成功',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='✅ 解除綁定成功！', weight='bold', size='xl', color='#00FF00', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text=f'ID 「{user_id}」 已解除綁定。', size='lg', margin='md', weight='bold', wrap=True)
                            ]
                        )
                    )
                )
            else:
                message = FlexSendMessage(
                    alt_text='解除綁定失敗',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='❌ 解除綁定失敗。', weight='bold', size='xl', color='#FF0000', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text='您輸入的ID未綁定到您的LINE帳戶。', size='lg', margin='md', weight='bold', wrap=True)
                            ]
                        )
                    )
                )
        else:
            message = FlexSendMessage(
                alt_text='格式錯誤',
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text='❌ 格式錯誤。', weight='bold', size='xl', color='#FF0000', wrap=True),
                            SeparatorComponent(margin='md'),
                            TextComponent(text='請使用：解除綁定 <您的ID>。', size='lg', margin='md', weight='bold', wrap=True)
                        ]
                    )
                )
            )

    elif text.startswith('查詢綁定'):
        parts = text.split()
        if len(parts) == 2:
            _, user_id = parts
            if user_id in user_map and user_map[user_id] == line_user_id:
                message = FlexSendMessage(
                    alt_text='綁定查詢',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='🔍 綁定查詢', weight='bold', size='xl', color='#0000FF', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text=f'ID 「{user_id}」 已綁定到您的LINE帳戶。', size='lg', margin='md', weight='bold', wrap=True)
                            ]
                        )
                    )
                )
            else:
                message = FlexSendMessage(
                    alt_text='查詢結果',
                    contents=BubbleContainer(
                        body=BoxComponent(
                            layout='vertical',
                            contents=[
                                TextComponent(text='❌ 查詢結果', weight='bold', size='xl', color='#FF0000', wrap=True),
                                SeparatorComponent(margin='md'),
                                TextComponent(text=f'ID 「{user_id}」 未綁定到您的LINE帳戶。', size='lg', margin='md', weight='bold', wrap=True)
                            ]
                        )
                    )
                )
        else:
            message = FlexSendMessage(
                alt_text='格式錯誤',
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text='❌ 格式錯誤。', weight='bold', size='xl', color='#FF0000', wrap=True),
                            SeparatorComponent(margin='md'),
                            TextComponent(text='請使用：查詢綁定 <您的ID>。', size='lg', margin='md', weight='bold', wrap=True)
                        ]
                    )
                )
            )

    elif text == '查看所有綁定':
        bound_ids = [uid for uid, line_id in user_map.items() if line_id == line_user_id]
        if bound_ids:
            message = FlexSendMessage(
                alt_text='已綁定的ID',
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text='📃 已綁定的ID：', weight='bold', size='xl', color='#0000FF', wrap=True),
                            SeparatorComponent(margin='md'),
                            TextComponent(text=', '.join(bound_ids), size='lg', margin='md', weight='bold', wrap=True)
                        ]
                    )
                )
            )
        else:
            message = FlexSendMessage(
                alt_text='無綁定ID',
                contents=BubbleContainer(
                    body=BoxComponent(
                        layout='vertical',
                        contents=[
                            TextComponent(text='📃 無綁定ID', weight='bold', size='xl', color='#FF0000', wrap=True),
                            SeparatorComponent(margin='md'),
                            TextComponent(text='您沒有任何已綁定的ID。', size='lg', margin='md', weight='bold', wrap=True)
                        ]
                    )
                )
            )

    elif text in ['help', '幫助', '指南', '使用方法']:
        message = FlexSendMessage(
            alt_text='操作指南',
            contents=BubbleContainer(
                body=BoxComponent(
                    layout='vertical',
                    contents=[
                        TextComponent(text='📌 操作指南：', weight='bold', size='xl', color='#0000FF', wrap=True),
                        SeparatorComponent(margin='md'),
                        TextComponent(text='1. 綁定您的ID：綁定 <您的ID>', size='lg', margin='md', weight='bold', wrap=True),
                        TextComponent(text='2. 更改您的ID：更改ID <舊的ID> <新的ID>', size='lg', margin='md', weight='bold', wrap=True),
                        TextComponent(text='3. 解除綁定：解除綁定 <您的ID>', size='lg', margin='md', weight='bold', wrap=True),
                        TextComponent(text='4. 查詢綁定：查詢綁定 <您的ID>', size='lg', margin='md', weight='bold', wrap=True),
                        TextComponent(text='5. 查看所有綁定：查看所有綁定', size='lg', margin='md', weight='bold', wrap=True)
                    ]
                )
            )
        )

    else:
        # 忽略未知指令，不做任何操作
        return

    line_bot_api.reply_message(event.reply_token, message)

if __name__ == "__main__":
    print("「血氧數據查詢平台」LINE BOT 連動通知程式啟動")
    app.run()