#include "BLEDevice.h"
#include <PubSubClient.h>
#include <WiFi.h>
#include <ArduinoJson.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>

#define SCREEN_WIDTH 128 // OLED顯示器寬度，以像素為單位
#define SCREEN_HEIGHT 64 // OLED顯示器高度，以像素為單位

// 宣告一個連接到I2C（SDA，SCL引腳）的SSD1306顯示器
#define OLED_RESET     -1 // 重置引腳號（如果共享Arduino重置引腳，則為-1）
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

#define MQTT_HOST "test.mosquitto.org"
#define MQTT_PORT 1883
#define MQTT_TOPIC "oximeter/json"

char ssid []    ="****";    // 輸入WiFi名稱
char password[] = "*****";  // 輸入WiFi密碼

void callback(char* topic, byte* payload, unsigned int length);
WiFiClient wifiClient;
PubSubClient mqtt(wifiClient);

#define UART_SERVICE_UUID      "CDEACB80-5235-4C07-8846-93A37EE6B86D"
#define CHARACTERISTIC_UUID_RX "CDEACB81-5235-4C07-8846-93A37EE6B86D"
#define CHARACTERISTIC_UUID_TX "CDEACB81-5235-4C07-8846-93A37EE6B86D"

#define STRING_BUF_SIZE 100

BLEAdvertData foundDevice;
BLEAdvertData targetDevice;
BLEClient* client;
BLERemoteService* UartService;
BLERemoteCharacteristic* Rx;
BLERemoteCharacteristic* Tx;

int heart_bit = 0;
int blood_oxygen = 0;
int PI_index = 0;
int id = 207;//ID修改

// 掃描回調函數
void scanCB(T_LE_CB_DATA* p_data) {
  foundDevice.parseScanInfo(p_data);
  if (foundDevice.hasName()) {
    if (foundDevice.getName() == String("Medical")) {
      Serial.print("發現Medical裝置在地址 ");
      Serial.println(foundDevice.getAddr().str());
      targetDevice = foundDevice;
    }
  }
}

// 通知回調函數
void notificationCB (BLERemoteCharacteristic* /*chr*/, uint8_t* data, uint16_t len) {
  byte msg[len + 1];
  memcpy(msg, data, len);

  if (msg[4] == 4) {
    heart_bit = msg[6]; blood_oxygen = msg[7]; PI_index = msg[8];
  }
}

// MQTT回調函數
void callback(char* topic, byte* payload, unsigned int length) {
  // 處理接收到的訊息
  Serial.print("消息發送於 [");
  Serial.print(topic);
  Serial.println("] ");
}

void setup() {
  Serial.begin(115200);
  WiFi.begin(ssid, password);
  
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }

  Serial.println("");
  Serial.println("WiFi已連接");  
  Serial.print("IP地址: ");
  Serial.println(WiFi.localIP());

  mqtt.setServer(MQTT_HOST, MQTT_PORT);
  mqtt.setCallback(callback);

  BLE.init();
  BLE.setScanCallback(scanCB);
  BLE.beginCentral(1);

  BLE.configScan()->startScan(3000);
  BLE.configConnection()->connect(targetDevice, 4000);
  delay(1000);
  int8_t connID = BLE.configConnection()->getConnId(targetDevice);
  if (!BLE.connected(connID)) {
    Serial.println("BLE未連接");
  } else {
    BLE.configClient();
    client = BLE.addClient(connID);
    client->discoverServices();
    Serial.print(">> 發現連接設備的服務");
    do {
      Serial.print(".");
      delay(1000);
    } while (!(client->discoveryDone()));
    Serial.println();

    UartService = client->getService(UART_SERVICE_UUID);
    if (UartService != nullptr) {
      Tx = UartService->getCharacteristic(CHARACTERISTIC_UUID_TX);
      if (Tx != nullptr) {
        Serial.println("找到TX特徵值");
        Tx->setBufferLen(STRING_BUF_SIZE);
        Tx->setNotifyCallback(notificationCB);
        Tx->enableNotifyIndicate();
      }
      Rx = UartService->getCharacteristic(CHARACTERISTIC_UUID_RX);
      if (Rx != nullptr) {
        Serial.println("找到RX特徵值");
        Rx->setBufferLen(STRING_BUF_SIZE);
      }
    }
  }

  connectToMqtt();
  // SSD1306_SWITCHCAPVCC = 從3.3V內部生成顯示電壓
  if (!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) { // 地址0x3D用於128x64
    Serial.println(F("SSD1306分配失敗"));
    for (;;); // 不繼續，無限循環
  }
  display.display();
  delay(2000); // 暫停2秒
  display.clearDisplay();
  display.display();
}

void loop() {
  if (Serial.available()) {
    Rx->writeString(Serial.readString());
  }
  mqtt.loop();

  if (!mqtt.connected()) {
    reconnectMqtt();
  }

  // 使用ArduinoJson構建JSON字串
  StaticJsonDocument<200> jsonDoc;
  JsonObject status = jsonDoc.createNestedObject("d");
  status["id"] = id;
  status["bpm"] = heart_bit;
  status["spo2"] = blood_oxygen;
  status["pi"] = PI_index * 0.1;

  char jsonString[200];
  serializeJson(jsonDoc, jsonString);
  Serial.println(jsonString);

  // 發佈MQTT訊息
  if (heart_bit > 0 && heart_bit < 255) {
    if (!mqtt.publish(MQTT_TOPIC, jsonString)) {
      Serial.println("MQTT發佈失敗");
    }
  }
  display.clearDisplay();
  display.setTextSize(2);
  display.setTextColor(WHITE);
  display.setCursor(0, 10);
  display.println("BPM");
  display.setCursor(60, 10);
  display.println(heart_bit);
  display.setCursor(0, 30);
  display.println("SPO2:");
  display.setCursor(60, 30);
  display.println(blood_oxygen);
  display.setCursor(0, 50);
  display.println("PI:");
  display.setCursor(60, 50);
  display.println(PI_index / 10.0);
  display.display();
  delay(3000); // 延遲3秒
}

// 連接到MQTT
void connectToMqtt() {
  while (!mqtt.connected()) {
    Serial.print("正在嘗試連接MQTT...");
    // 嘗試連接
    if (mqtt.connect(MQTT_DEVICEID)) {
      Serial.println("MQTT已連接");
      mqtt.subscribe(MQTT_TOPIC);
    } else {
      Serial.print("MQTT連接失敗，rc=");
      Serial.print(mqtt.state());
      Serial.println(" 5秒後重試");
      delay(5000);
    }
  }
}

// 重新連接MQTT
void reconnectMqtt() {
  // 直到重新連接為止
  while (!mqtt.connected()) {
    Serial.print("正在嘗試重新連接MQTT...");
    // 嘗試重新連接
    if (mqtt.connect(MQTT_DEVICEID)) {
      Serial.println("MQTT已重新連接");
      mqtt.subscribe(MQTT_TOPIC);
    } else {
      Serial.print("MQTT重新連接失敗，rc=");
      Serial.print(mqtt.state());
      Serial.println(" 5秒後重試");
      delay(5000);
    }
  }
}
